/********************************************************************************;/
*                                                                               *
*                                MightySETIInfo                                 *
*                                                                               *
*********************************************************************************
* Copyright (C) 2002 by Drake Christensen.   All Rights Reserved.               *
*********************************************************************************
* $Id: MightySETIInfo.cpp,v 1.00 2002/09/18 03:30:00 mighty Exp $               *
********************************************************************************/
#include "fx.h"
#include <stdio.h>
#include <stdlib.h>
#include <sys/stat.h>
#ifndef WIN32
#include <unistd.h>
#endif
#include <time.h>

// A horrible hack to fight off-by-one-hour bug in MSVC during daylight stupid time Q182042 Q190315
#define BUG_DST 1
#if BUG_DST
#ifdef __cplusplus
extern "C" {
#endif  /* __cplusplus */
extern int __cdecl _isindst(struct tm *);
#ifdef __cplusplus
}
#endif  /* __cplusplus */
#endif

#include "FXString.h"
#include "FXSettings.h"
#include "FXSplitter.h"
#include "FXrex.h"

#include "SETIConsts.h"

#include "SETIFTP.h"

#include "SETISettings.h"
#include "SETIFTPSettings.h"
#include "SETIText.h"
#include "FTPConfigDlg.h"
#include "DebugFlags.h"
#include "DebugDlg.h"
#include "MainConfigDlg.h"
#include "FTPProgressDlg.h"

#include "SETIInfo.h"

#include "MightySETIInfoApp.h"
#include "MightySETIInfo.h"

#define DEBUG_REPLACEMENT 0

//---------------------------------------------------------------------------
const FXchar k_arrcAppTitle[] = "Mighty's SETI@Home Info";
const FXuint k_uVersionMajor = 4;
const FXuint k_uVersionMinor = 0;
const FXchar k_arrcVersionModifier[] = "";
#define k_SettingsFilename "MightySETIInfo.ini"
const FXchar k_arrcUserInfo[] = "C:\\Program Files\\SETI@Home";
const FXchar k_arrcDefaultSigInputName[] = "MightySETIInfoSigInput.txt";
const FXchar k_arrcDefaultSigName[] = "MightySETIInfoSig.txt";
const FXbool k_fDefaultSigEnabled = 1;
const FXint k_nDefaultSigInterval = 86400;	// One day
const FXchar k_arrcDefaultSigLastWritten[] = "2003-01-01 00:00:00";
const FXbool k_fDefaultUpdateOnceThenExit = 1;
const FXbool k_fDefaultUsingSetiHide=0;
const FXchar k_arrcDefaultHTMLInputName[] = "MightySETIInfoInput.html";
const FXchar k_arrcDefaultHTMLName[] = "MightySETIInfo.html";
const FXbool k_fDefaultHTMLEnabled = 1;
const FXint k_nDefaultHTMLInterval = 300;	// Five mins
const FXchar k_arrcDefaultHTMLLastWritten[] = "2003-01-01 00:00:00";
const FXbool k_fDefaultFTPEnabled = 0;
const FXint k_nDefaultFTPFirewall = 0;
const FXchar k_arrcDefaultFTPServer[] = "ftp.MightySETIInfo.com";
const FXchar k_arrcDefaultFTPUsername[] = "MightySETIInfoUsername";
const FXchar k_arrcDefaultFTPPassword[] = "MightySETIInfoPassword";
const FXchar k_arrcDefaultFTPDirectory[] = "MightySETIInfoDirectory";
const FXchar k_arrcDefaultFTPFirewallServer[] = "ftp.MightySETIInfo.com";
const FXchar k_arrcDefaultFTPFirewallUsername[] = "MightySETIInfoUsername";
const FXchar k_arrcDefaultFTPFirewallPassword[] = "MightySETIInfoPassword";
const FXbool k_fDefaultFTPPassive = 1;
const FXchar k_arrcDefaultQuote[] = "MightySETIInfoQuotes.txt";
const FXchar k_arrcQuoteDelimiter[] = "\\n\\n";
const FXint k_nQuoteDelimiterAdjustFront = 0;
const FXint k_nQuoteDelimiterAdjustBack = 0;
// For AddS()
const FXchar k_arrcS[] = "s";
const FXchar k_arrcEmpty[] = "\0";

static const unsigned char s_infoIcon[]={
0x47,0x49,0x46,0x38,0x37,0x61,0x20,0x00,0x20,0x00,0xf2,0x00,0x00,0x80,0x80,0x80,
0xc0,0xc0,0xc0,0xff,0xff,0xff,0x00,0x00,0xff,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x2c,0x00,0x00,0x00,0x00,0x20,0x00,0x20,0x00,0x00,0x03,
0x8c,0x08,0xba,0xdc,0xfe,0xf0,0x05,0x41,0x6b,0x88,0x58,0xcd,0xca,0xf9,0xcd,0xcc,
0xd6,0x8d,0x16,0x08,0x90,0xc1,0xa0,0x0e,0x22,0x47,0x60,0xa4,0xb0,0xae,0xf1,0xfb,
0xc4,0xf2,0x3c,0xe0,0x76,0x88,0xa7,0xab,0x56,0x27,0xd0,0x53,0xe0,0x8e,0xa4,0xa2,
0x10,0x79,0x24,0x2e,0x8e,0x3a,0x15,0xb2,0xc7,0xd4,0x4d,0x9f,0x48,0xeb,0x91,0x9a,
0x9d,0x5d,0x8d,0x5d,0xda,0x76,0xb1,0xec,0x68,0x63,0x4e,0x2c,0xee,0x9c,0x6c,0x94,
0x2b,0xec,0x61,0x11,0x3c,0x8a,0x8a,0x3b,0x73,0x05,0x81,0x39,0x86,0xec,0xf9,0x23,
0x79,0x0d,0x04,0x6f,0x68,0x82,0x0e,0x04,0x84,0x4c,0x44,0x87,0x7e,0x89,0x4b,0x8c,
0x8d,0x20,0x89,0x8a,0x14,0x92,0x26,0x0c,0x7f,0x02,0x97,0x98,0x7a,0x15,0x9c,0x9d,
0x00,0x7f,0xa0,0xa1,0x7b,0xa4,0xa5,0xa7,0xa1,0xaa,0xab,0x19,0x09,0x00,0x3b
};

//---------------------------------------------------------------------------
// Map
FXDEFMAP(MightySETIInfoWindow) MightySETIInfoWindowMap[]={
  FXMAPFUNC(SEL_COMMAND, MightySETIInfoWindow::ID_ABOUT, MightySETIInfoWindow::onCmdAbout),
  FXMAPFUNC(SEL_COMMAND, MightySETIInfoWindow::ID_BTNDOIT, MightySETIInfoWindow::onCmdDoIt),
  FXMAPFUNC(SEL_COMMAND, MightySETIInfoWindow::ID_BTNCONFIGURE, MightySETIInfoWindow::onCmdShowMainConfig),
  FXMAPFUNC(SEL_COMMAND, MightySETIInfoWindow::ID_CHKONCETHENEXIT, MightySETIInfoWindow::onCmdOnceThenExit),
  FXMAPFUNC(SEL_UPDATE, MightySETIInfoWindow::ID_CHKONCETHENEXIT, MightySETIInfoWindow::onUpdOnceThenExit),
  FXMAPFUNC(SEL_COMMAND, MightySETIInfoWindow::ID_SETUSERINFO, MightySETIInfoWindow::onCmdBrowseUserInfo),
  FXMAPFUNC(SEL_COMMAND, MightySETIInfoWindow::ID_SETSIGINPUT, MightySETIInfoWindow::onCmdBrowseSigIn),
  FXMAPFUNC(SEL_COMMAND, MightySETIInfoWindow::ID_SETSIGOUTPUT, MightySETIInfoWindow::onCmdBrowseSig),
  FXMAPFUNC(SEL_COMMAND, MightySETIInfoWindow::ID_SETHTMLINPUT, MightySETIInfoWindow::onCmdBrowseHTMLIn),
  FXMAPFUNC(SEL_COMMAND, MightySETIInfoWindow::ID_SETHTMLOUTPUT, MightySETIInfoWindow::onCmdBrowseHTML),
  FXMAPFUNC(SEL_COMMAND, MightySETIInfoWindow::ID_SHOWMAINCONFIG, MightySETIInfoWindow::onCmdShowMainConfig),
  FXMAPFUNC(SEL_COMMAND, MightySETIInfoWindow::ID_SHOWFTP, MightySETIInfoWindow::onCmdShowFTP),
  FXMAPFUNC(SEL_COMMAND, MightySETIInfoWindow::ID_SHOWDEBUG, MightySETIInfoWindow::onCmdShowDebug),
  FXMAPFUNC(SEL_CHORE,MightySETIInfoWindow::ID_IDLETASKSTARTUP,MightySETIInfoWindow::onIdleTaskStartup),
  FXMAPFUNC(SEL_CHORE,MightySETIInfoWindow::ID_FTPCHORE,MightySETIInfoWindow::onFTPChore),
  FXMAPFUNC(SEL_TIMEOUT,MightySETIInfoWindow::ID_TIMERTOCONTINUE,MightySETIInfoWindow::onTimerContinue),
  FXMAPFUNC(SEL_TIMEOUT,MightySETIInfoWindow::ID_TIMERTOSHUTDOWN,MightySETIInfoWindow::onTimerShutdown),
  FXMAPFUNC(SEL_TIMEOUT,MightySETIInfoWindow::ID_TIMERHEARTBEAT,MightySETIInfoWindow::onTimerHeartbeat),
  };

//---------------------------------------------------------------------------
// Object implementation
FXIMPLEMENT(MightySETIInfoWindow,FXMainWindow,MightySETIInfoWindowMap,ARRAYNUMBER(MightySETIInfoWindowMap))

//---------------------------------------------------------------------------
// Make some windows
MightySETIInfoWindow::MightySETIInfoWindow(FXApp* a):FXMainWindow(a,k_arrcAppTitle,NULL,NULL,
		DECOR_ALL & ~(DECOR_MAXIMIZE | DECOR_RESIZE),0,0,650,650),
		menubar(NULL),
		filemenu(NULL),
		m_menuConfigure(NULL),
		m_menuDebug(NULL),
		helpmenu(NULL),
		m_chkOnceThenExit(NULL),
		m_btnConfigure(NULL),
		m_btnDoIt(NULL),
		m_txtSig(NULL),
		m_txtHTML(NULL),
		m_grpboxSig(NULL),
		m_grpboxHTML(NULL),
		m_chkEnableSig(NULL),
		m_chkEnableHTML(NULL),
		m_chkEnableFTP(NULL),
		m_uDelayBeforeExit(10),
		m_ptimerHeartbeat(NULL),
		m_pDebugFlags(NULL),
		m_pdialogDebug(NULL),
		m_dlgFTPContinue(NULL),
		m_dlgFTPProgress(NULL)
{
	m_info.m_fSuccessfullyParsed = FALSE;
	m_info.m_strFilename = "";
	m_info.m_strName = "";
	m_info.m_strID = "";
	m_info.m_rTotalSecs = 0.0;
	m_info.m_uYears = 0;
	m_info.m_uDays = 0;
	m_info.m_uHours = 0;
	m_info.m_uMins = 0;
	m_info.m_uSecs = 0;
	m_info.m_uWUs = 0;
	time( &(m_info.m_time) );
	srand( (unsigned)time( NULL ) );	// For now, always random

	m_FTPSettings.m_fEnabled = FALSE;
	m_FTPSettings.m_nFirewallIndex = 0;
	m_FTPSettings.m_fPassive = FALSE;

	// Make menu bar
	menubar=new FXMenubar(this,LAYOUT_FILL_X);
	filemenu=new FXMenuPane(this);
	new FXMenuCommand(filemenu,"&Quit\tCtl-Q",NULL,getApp(),FXApp::ID_QUIT);
	new FXMenuTitle(menubar,"&File",NULL,filemenu);
	// Configure
	m_menuConfigure=new FXMenuPane(this);
	new FXMenuCommand(m_menuConfigure,"&Main Settings\tCtl-M",NULL,this,ID_SHOWMAINCONFIG);
	new FXMenuCommand(m_menuConfigure,"&Root directory\tCtl-R",NULL,this,ID_SETUSERINFO);
	new FXMenuCommand(m_menuConfigure,"&Input sig file\tCtl-I",NULL,this,ID_SETSIGINPUT);
	new FXMenuCommand(m_menuConfigure,"&Output sig file\tCtl-O",NULL,this,ID_SETSIGOUTPUT);
	new FXMenuCommand(m_menuConfigure,"I&nput HTML file\tCtl-N",NULL,this,ID_SETHTMLINPUT);
	new FXMenuCommand(m_menuConfigure,"Output &HTML file\tCtl-H",NULL,this,ID_SETHTMLOUTPUT);
	new FXMenuCommand(m_menuConfigure,"&FTP Settings\tCtl-F",NULL,this,ID_SHOWFTP);
	new FXMenuTitle(menubar,"&Configure",NULL,m_menuConfigure);
	// Debug
	m_menuDebug=new FXMenuPane(this);
	new FXMenuCommand(m_menuDebug,"&Show\tCtl-S",NULL,this,ID_SHOWDEBUG);
	new FXMenuTitle(menubar,"&Debug",NULL,m_menuDebug);
	// Help
	helpmenu=new FXMenuPane(this);
	new FXMenuCommand(helpmenu,"&About MightySETIInfo...",NULL,this,ID_ABOUT,0);
	new FXMenuTitle(menubar,"&Help",NULL,helpmenu);

	FXVerticalFrame * l_frameMain = new FXVerticalFrame(this,LAYOUT_FILL_X|LAYOUT_FILL_Y,0,0,0,0, 0,0,0,0);

	FXHorizontalFrame * l_frameHoriz = new FXHorizontalFrame(l_frameMain, FRAME_LINE|LAYOUT_FILL_X,0,0,0,0, 0,0,0,0);
	m_btnConfigure = new FXButton(l_frameHoriz, " Main Settings \t\tConfigure settings", NULL, this, ID_BTNCONFIGURE, BUTTON_NORMAL | LAYOUT_LEFT);
	m_btnConfigOutputFTP = new FXButton(l_frameHoriz, " FTP Configure \t\tDefine FTP server", NULL, this, ID_BTNOUTPUTFTP);
	m_chkOnceThenExit = new FXCheckButton(l_frameHoriz, "Update Once then exit\t\tAfter writing the files, wait a few seconds then quit", this, ID_CHKONCETHENEXIT);
	m_btnDoIt = new FXButton(l_frameHoriz, "  Do It  \t\tAfter making changes, rewrite files", NULL, this, ID_BTNDOIT, BUTTON_NORMAL | LAYOUT_RIGHT);

	// Splitter window between output boxes
	FXSplitter *l_splitter=new FXSplitter(l_frameMain,LAYOUT_SIDE_TOP|LAYOUT_FILL_X|LAYOUT_FILL_Y|SPLITTER_VERTICAL|SPLITTER_TRACKING);
	// Make m_txtSig
	FXGroupBox * l_boxTemp = new FXGroupBox(l_splitter, "Sig", GROUPBOX_TITLE_LEFT|FRAME_RIDGE|LAYOUT_FILL_X|LAYOUT_FILL_Y, 0,0,0,240);
	FXVerticalFrame * l_frameTemp = new FXVerticalFrame(l_boxTemp, FRAME_LINE|LAYOUT_FILL_X|LAYOUT_FILL_Y,0,0,0,0, 0,0,0,0);
	m_txtSig=new SETIText(l_frameTemp);
	m_txtSig->setLayoutHints(LAYOUT_FILL_X | LAYOUT_FILL_Y);
	m_txtSig->setEditable(FALSE);
	m_grpboxSig = l_boxTemp;

	// Make m_txtHTML
	l_boxTemp = new FXGroupBox(l_splitter, "HTML", GROUPBOX_TITLE_LEFT|FRAME_RIDGE|LAYOUT_FILL_X|LAYOUT_FILL_Y);
	l_frameTemp = new FXVerticalFrame(l_boxTemp, FRAME_LINE|LAYOUT_FILL_X|LAYOUT_FILL_Y,0,0,0,0, 0,0,0,0);
	m_txtHTML=new SETIText(l_frameTemp);
	m_txtHTML->setLayoutHints(LAYOUT_FILL_X | LAYOUT_FILL_Y);
	m_txtHTML->setEditable(FALSE);
	m_grpboxHTML = l_boxTemp;

	// Order of operations detail.  Since we log what we read from the INI file
	// we need the debug dialog instantiated to take those log entries.  Instead,
	// I should change the Log() function to store off the string and use the
	// update message in Fox to load the editbox
	m_pdialogDebug = new DebugDialog(this);

	FXString l_strTemp;
	l_strTemp.format("Parsing main settings file [%s]", k_SettingsFilename);
	Log(l_strTemp);
	FXbool l_fFileParsed = m_MainSettings.parseFile( k_SettingsFilename, TRUE );
	if ( !l_fFileParsed)
		{
		// TODO: Set up an error reporting system that lets me defer the reporting
		// of this warning "No INI file found.  Using default settings"
		// We can't display a dialog box right now because the main window hasn't
		// been created yet
		WriteDefaultINI();
		}

	m_pDebugFlags = new DebugFlags(&m_MainSettings);
	m_pdialogDebug->ShowFlags();

	FXint l_fEnabled = m_MainSettings.readIntEntry("Sig", "Enabled", TRUE);

	GetFTPSettings(&m_FTPSettings);

	// Register Chore callback message to read setup files
	// One reason to do it this way is so we have a main window up and running
	// if we encounter any errors while reading/writing
	FXChore* chorehandle = getApp()->addChore(this,ID_IDLETASKSTARTUP);
}

//---------------------------------------------------------------------------
MightySETIInfoWindow::~MightySETIInfoWindow()
{
  m_pDebugFlags->WriteFlags(&m_MainSettings);
  WriteINI();
  delete m_pDebugFlags;
  delete filemenu;
  delete helpmenu;
  delete m_menuConfigure;
  delete m_menuDebug;
  delete m_pdialogDebug;
  }

//---------------------------------------------------------------------------
// Start
void MightySETIInfoWindow::create()
{
  FXMainWindow::create();
  show(PLACEMENT_SCREEN);
  }

//---------------------------------------------------------------------------
// About
long MightySETIInfoWindow::onCmdAbout(FXObject*,FXSelector,void*)
{
FXString l_strText;
l_strText.format("Copyright  2002, 2003 All Rights Reserved Drake A. Christensen\n\nA toy project to keep my skills up.\n\nWrites a text file with the number of work units\nand time spent by SETI@Home on this machine\n\nVersion %s\n\nhttp://www.mightydrake.com/SetiAtHome/MightySETIInfo/",
		GetProgramVersion().text());
FXMessageBox::information(this,MBOX_OK,"About MightySETIInfo",l_strText.text());
return 1;
}

//---------------------------------------------------------------------------
// Enable/disable Once then exit
long MightySETIInfoWindow::onCmdOnceThenExit(FXObject*,FXSelector,void*)
{
	Log("onCmdOnceThenExit");
	// Toggle it here
	FXint l_fOnceThenExit = m_chkOnceThenExit->getCheck();
	m_MainSettings.writeIntEntry("System", "UpdateOnceThenExit", l_fOnceThenExit);
	WriteINI();
	return 1;
}

//---------------------------------------------------------------------------
// Update check state of Once then exit
long MightySETIInfoWindow::onUpdOnceThenExit(FXObject*,FXSelector,void*)
{
	FXint l_fOnceThenExit = m_MainSettings.readIntEntry("System", "UpdateOnceThenExit", k_fDefaultUpdateOnceThenExit);
	m_chkOnceThenExit->setCheck(l_fOnceThenExit);
	return 1;
}

//---------------------------------------------------------------------------
// Choose input file
long MightySETIInfoWindow::onCmdDoIt(FXObject*,FXSelector,void*)
{
	Log("User chose to do it");
	ClearBoxes();
	DoIt();
	return 1;
}

//---------------------------------------------------------------------------
long MightySETIInfoWindow::onCmdBrowseUserInfo(FXObject*,FXSelector,void*)
{
	FXString l_strTemp;
	Log("onCmdBrowseUserInfo");
	FXString l_strFile;
	l_strTemp = m_MainSettings.readStringEntry("Machines", "Root", k_arrcUserInfo);
	FXDirDialog l_dir(this, l_strTemp);
	l_dir.setDirectory(l_strTemp);
	if ( l_dir.execute() )
		{
		l_strFile=l_dir.getDirectory();
		m_MainSettings.writeStringEntry("Machines", "Root", l_strFile.text());
		l_strTemp.format("Writing main settings file [%s]", k_SettingsFilename);
		Log(l_strTemp);
		WriteINI();
		}
	// TODO: Should really test to see if we're not using Setihide.
	// Because if not, then there should be a user_info here
	return 1;
}

//---------------------------------------------------------------------------
// Choose output file
long MightySETIInfoWindow::onCmdBrowseSigIn(FXObject*,FXSelector,void*)
{
	FXString l_strTemp;
	Log("onCmdBrowseSigIn");
	FXString l_strFile;
	FXFileDialog opendialog(this,"Choose input sig file");
	opendialog.setSelectMode( SELECTFILE_ANY );
	opendialog.setPatternList("*.txt,*.sig,*.*");
	opendialog.setFilename( k_arrcDefaultSigInputName );
	if(opendialog.execute())
		{
		l_strFile=opendialog.getFilename();
		m_MainSettings.writeStringEntry("Sig", "Input", l_strFile.text());
		l_strTemp.format("Writing main settings file [%s]", k_SettingsFilename);
		Log(l_strTemp);
		WriteINI();
		}
	return 1;
}

//---------------------------------------------------------------------------
// Choose output file
long MightySETIInfoWindow::onCmdBrowseSig(FXObject*,FXSelector,void*)
{
	FXString l_strTemp;
	Log("onCmdBrowseSig");
	FXString l_strFile;
	FXFileDialog opendialog(this,"Choose output sig file");
	opendialog.setSelectMode( SELECTFILE_ANY );
	opendialog.setPatternList("*.txt,*.sig,*.*");
	opendialog.setFilename( k_arrcDefaultSigName );
	if(opendialog.execute())
		{
		l_strFile=opendialog.getFilename();
		m_MainSettings.writeStringEntry("Sig", "Output", l_strFile.text());
		l_strTemp.format("Writing main settings file [%s]", k_SettingsFilename);
		Log(l_strTemp);
		WriteINI();
		}
	return 1;
}

//---------------------------------------------------------------------------
// Choose input file
long MightySETIInfoWindow::onCmdBrowseHTMLIn(FXObject*,FXSelector,void*)
{
	FXString l_strTemp;
	Log("onCmdHTMLIn");
	FXString l_strFile;
	FXFileDialog opendialog(this,"Choose input HTML file");
	opendialog.setSelectMode( SELECTFILE_ANY );
	opendialog.setPatternList("*.htm,*.html,*.*");
	opendialog.setFilename( k_arrcDefaultHTMLInputName );
	if(opendialog.execute())
		{
		l_strFile=opendialog.getFilename();
		m_MainSettings.writeStringEntry("HTML", "Input", l_strFile.text());
		l_strTemp.format("Writing main settings file [%s]", k_SettingsFilename);
		Log(l_strTemp);
		WriteINI();
		}
	return 1;
}

//---------------------------------------------------------------------------
// Choose output file
long MightySETIInfoWindow::onCmdBrowseHTML(FXObject*,FXSelector,void*)
{
	FXString l_strTemp;
	Log("onCmdHTML");
	FXString l_strFile;
	FXFileDialog opendialog(this,"Choose output HTML file");
	opendialog.setSelectMode( SELECTFILE_ANY );
	opendialog.setPatternList("*.htm,*.html,*.*");
	opendialog.setFilename( k_arrcDefaultHTMLName );
	if(opendialog.execute())
		{
		l_strFile=opendialog.getFilename();
		m_MainSettings.writeStringEntry("HTML", "Output", l_strFile.text());
		l_strTemp.format("Writing main settings file [%s]", k_SettingsFilename);
		Log(l_strTemp);
		WriteINI();
		}
	return 1;
}

//---------------------------------------------------------------------------
long MightySETIInfoWindow::onIdleTaskStartup(FXObject*,FXSelector,void*)
{
	FXbool l_fOK = ReadSETIFiles(GetInputFilename());
	if ( l_fOK )
		{
		DoItFirstTime();
		}
	return 1;
}

//---------------------------------------------------------------------------
long MightySETIInfoWindow::onFTPChore(FXObject*,FXSelector,void*)
{
	DoFTPChore();
	return 1;
}

//---------------------------------------------------------------------------
long MightySETIInfoWindow::onTimerContinue(FXObject*,FXSelector sel,void*)
{
	getApp()->stopModal(m_dlgFTPContinue,
			MBOX_CLICKED_YES+(SELID(FXMessageBox::ID_CLICKED_OK)-FXMessageBox::ID_CLICKED_YES));
 	delete m_dlgFTPContinue;
	m_dlgFTPContinue = NULL;
	Defibrillate();
	return 1;
}

//---------------------------------------------------------------------------
long MightySETIInfoWindow::onTimerShutdown(FXObject*,FXSelector,void*)
{
	getApp()->handle(this, MKUINT(FXApp::ID_QUIT, SEL_COMMAND), NULL);
	return 1;
}

//---------------------------------------------------------------------------
long MightySETIInfoWindow::onTimerHeartbeat(FXObject*,FXSelector,void*)
{
	m_ptimerHeartbeat = NULL;
	Heartbeat();
	return 1;
}

//---------------------------------------------------------------------------
long MightySETIInfoWindow::onCmdShowMainConfig(FXObject*,FXSelector,void*)
{
	MainConfigDialog *l_pdialog = new MainConfigDialog(this, &m_MainSettings,
			&m_FTPSettings, m_pdialogDebug);
	if (l_pdialog->execute(PLACEMENT_OWNER))
		{
		SaveFTPSettings(&m_FTPSettings);
		WriteINI();
		}
	delete l_pdialog;
	l_pdialog = NULL;
	return 1;
}

//---------------------------------------------------------------------------
long MightySETIInfoWindow::onCmdShowFTP(FXObject*,FXSelector,void*)
{
	FTPConfigDialog *l_pdialogFTP = new FTPConfigDialog(this, &m_FTPSettings);
	if (l_pdialogFTP->execute(PLACEMENT_OWNER))
		{
		SaveFTPSettings(&m_FTPSettings);
		WriteINI();
		}
	delete l_pdialogFTP;
	l_pdialogFTP = NULL;
	return 1;
}

//---------------------------------------------------------------------------
long MightySETIInfoWindow::onCmdShowDebug(FXObject*,FXSelector,void*)
{
	m_pdialogDebug->show();
	return 1;
}

//---------------------------------------------------------------------------
FXString MightySETIInfoWindow::GetInputFilename()
{
FXString l_strFilename;
// Get root dir from INI file
FXString l_strRootDir = m_MainSettings.readStringEntry("Machines", "Root", k_arrcUserInfo);
FXString l_strTemp;
l_strTemp.format("Machines | Root=[%s]\n", l_strRootDir.text());
appendPrefs(l_strTemp);
const FXString FXPathSep = PATHSEPSTRING;
const FXString lk_strUserInfo = FXPathSep + "user_info.sah";

FXbool l_fSetihide = m_MainSettings.readIntEntry("System", "UsingSetiHide", k_fDefaultUsingSetiHide);
if ( l_fSetihide )
	{
	FXString * l_strlistDirs = NULL;
	FXint l_nCount = FXFile::listFiles(l_strlistDirs, l_strRootDir, "wu_cache_*", LIST_MATCHING_DIRS);
	if ( l_nCount )
		{
		FXTime l_timeOldest = 0;
		FXString l_strOldestFilename = "";
		for (int i=0; i<l_nCount; i++)
			{
			FXString l_strCurFilename = l_strRootDir + FXPathSep + l_strlistDirs[i] + lk_strUserInfo;
			FXTime l_time = FXFile::modified(l_strCurFilename);
			if ( DBGFLG("SetihideSearch") )
				{
				l_strTemp.format("Size test=[%s]=%d, modified=%d", l_strCurFilename.text(),
					FXFile::size(l_strCurFilename), FXFile::modified(l_strCurFilename));
				Log(l_strTemp);
				}
			if ( l_time > l_timeOldest )
				{
				l_timeOldest = l_time;
				l_strOldestFilename = l_strRootDir + FXPathSep + l_strlistDirs[i];
				}
			}
		l_strRootDir = l_strOldestFilename;
		}
	}
// else, Root points to the SETI@Home directory and we can just append the filename

// Assume no trailing slash
l_strFilename = l_strRootDir + lk_strUserInfo;
l_strTemp.format("Input filename=[%s]\n", l_strFilename.text());
Log(l_strTemp);

return l_strFilename;
}

//---------------------------------------------------------------------------
FXbool MightySETIInfoWindow::ReadSETIFiles(FXString i_strFilename)
{
	FXbool l_fOK = FALSE;
	FXString l_strTemp;

	// Save off filename
	m_info.m_strFilename = i_strFilename;

	SETISettings settings;
	l_strTemp.format("Parsing input file [%s]", m_info.m_strFilename.text());
	Log(l_strTemp);
	FXbool l_fParsed = settings.parseFile(m_info.m_strFilename, FALSE);
	if ( l_fParsed )
		{
		// Read user info
		m_info.m_strID = settings.readStringEntry("SETIEmpty", "id");
		m_info.m_strName = settings.readStringEntry("SETIEmpty", "name");
		m_info.m_rTotalSecs = (float)(settings.readRealEntry("SETIEmpty", "total_cpu"));
		m_info.m_uWUs = settings.readIntEntry("SETIEmpty", "nresults");

		// Calculate commonly-used values
		const FXdouble lk_rAHour = 3600;
		const FXdouble lk_rADay = lk_rAHour * 24.;
		const FXdouble lk_rAYear = lk_rADay * 365.24;
		m_info.m_uYears = (FXuint)(m_info.m_rTotalSecs / lk_rAYear);
		m_info.m_uDays = (FXuint)((m_info.m_rTotalSecs - (m_info.m_uYears * lk_rAYear)) / lk_rADay);
		m_info.m_uHours = (FXuint)((m_info.m_rTotalSecs - ((m_info.m_uYears * lk_rAYear) + (m_info.m_uDays * lk_rADay))) / lk_rAHour);
		m_info.m_uMins = (FXuint)((m_info.m_rTotalSecs -
			((m_info.m_uYears * lk_rAYear) + (m_info.m_uDays * lk_rADay) + (m_info.m_uHours * lk_rAHour))) / 60.);
		m_info.m_uSecs = (FXuint)(m_info.m_rTotalSecs -
			((m_info.m_uYears * lk_rAYear) + (m_info.m_uDays * lk_rADay) + (m_info.m_uHours * lk_rAHour) + (m_info.m_uMins * 60.)));

		// Log some stuff
		l_strTemp.format("id=[%s]\n", m_info.m_strID.text());
		appendUserInfo(l_strTemp);
		l_strTemp.format("name=[%s]\n", m_info.m_strName.text());
		appendUserInfo(l_strTemp);
		l_strTemp.format("taotal_cpu=%f\n", m_info.m_rTotalSecs);
		appendUserInfo(l_strTemp);
		l_strTemp.format("nresults=%d\n", m_info.m_uWUs);
		appendUserInfo(l_strTemp);
		l_fOK = TRUE;
		}
	else
		{
		FXMessageBox::error(this,MBOX_OK,"Error user info file","Error parsing file: [%s]", m_info.m_strFilename.text());
		}
	return l_fOK;
}

//---------------------------------------------------------------------------
void MightySETIInfoWindow::ReplaceString(FXString *i_strInput, const FXString ik_strField, const FXString ik_strReplacement)
{
	if ( DBGFLG("ReplaceString") )
		Log(FXString("ReplaceString( ") + ik_strField + FXString(", ") + ik_strReplacement + FXString(")"));
	FXString l_strTemp = FXString("<%") + ik_strField + FXString("%>");
	FXRex l_rex;
	FXRexError l_rexErr;
	FXint const lk_iComparisons = 1;
	l_rexErr = l_rex.parse(l_strTemp, REX_ICASE);
	if ( REGERR_OK == l_rexErr )
		{
		FXbool l_fOK = TRUE;
		while ( l_fOK )
			{
			FXint l_nBeginning = 0;
			FXint l_nEnd = 0;
			l_fOK = l_rex.match(*i_strInput, &l_nBeginning, &l_nEnd, REX_FORWARD, lk_iComparisons);
			if ( l_fOK )
				{
				if ( DBGFLG("ReplaceString") )
					Log("Found");
				(*i_strInput).replace(l_nBeginning, l_nEnd - l_nBeginning, ik_strReplacement);
				}
			else
				{
				if ( DBGFLG("ReplaceString") )
					Log("Not Found");
				}
			}
		}
	else
		{
		Log(FXRex::getError(l_rexErr));
		}
}

//---------------------------------------------------------------------------
void MightySETIInfoWindow::ReplaceInt(FXString *i_strInput, const FXString ik_strField, const FXuint ik_uInt,
	FXString const ik_strLabel)
{
	FXString l_strTemp;
	if ( ik_strLabel == FXString(""))
		{
		// Don't add S if no label
		l_strTemp.format("%d", ik_uInt);
		}
	else
		{
		l_strTemp.format("%d %s%s", ik_uInt, ik_strLabel.text(), AddS(ik_uInt));
		}
	ReplaceString(i_strInput, ik_strField, l_strTemp);
}

//---------------------------------------------------------------------------
void MightySETIInfoWindow::ReplaceFloat(FXString *i_strInput, const FXString ik_strField, const FXdouble ik_rNum,
	FXString const ik_strLabel)
{
	FXString l_strTemp;
	l_strTemp.format("%f %s", ik_rNum, ik_strLabel.text());
	ReplaceString(i_strInput, ik_strField, l_strTemp);
}

//---------------------------------------------------------------------------
FXbool MightySETIInfoWindow::DoReplacements(const FXString ik_strFilename, FXString * i_strTemplate)
{
	FXbool l_fOK = FALSE;
	FILE * fp=fopen(ik_strFilename.text(),"rb");
	if(fp)
		{
		// Read input file
		FXuint l_iSize=FXFile::size(ik_strFilename);

		// Make buffer to load file
		FXchar *l_sTemplate;
		FXbool l_fAlloced = FXCALLOC(&l_sTemplate,FXchar,l_iSize+1);
		if (l_fAlloced)
			{
			// Read the file
			FXuint n=fread(l_sTemplate,1,l_iSize,fp);
			// Close file
			fclose(fp);
			// Because these files are so small, I'm going to strip the <CR>s by reopening the file in text mode.  It's
			// incredibly ugly and brute force, but the simplest in code.  And the time shouldn't be enough to matter
			memset(l_sTemplate, 0, l_iSize);
			fp=fopen(ik_strFilename.text(),"r");
			fread(l_sTemplate,1,l_iSize,fp);
			fclose(fp);

			if ( n == l_iSize)
				{
				l_fOK = TRUE;

				// Put template in FXString
				*i_strTemplate = l_sTemplate;

				// Do the replacements
				ReplaceString( i_strTemplate, "User", m_info.m_strName);
				ReplaceFloat( i_strTemplate, "TotalSecsLbl", m_info.m_rTotalSecs, "seconds");
				ReplaceInt( i_strTemplate, "TotalSecsIntLbl", (FXint)(m_info.m_rTotalSecs), "second");
				ReplaceInt( i_strTemplate, "YearsLbl", m_info.m_uYears, "year");
				ReplaceInt( i_strTemplate, "DaysLbl", m_info.m_uDays, "day");
				ReplaceInt( i_strTemplate, "HoursLbl", m_info.m_uHours, "hour");
				ReplaceInt( i_strTemplate, "MinsLbl", m_info.m_uMins, "minute");
				ReplaceInt( i_strTemplate, "SecsLbl", m_info.m_uSecs, "second");
				ReplaceInt( i_strTemplate, "WUsLbl", m_info.m_uWUs, "work unit");
				ReplaceFloat( i_strTemplate, "TotalSecs", m_info.m_rTotalSecs);
				ReplaceInt( i_strTemplate, "TotalSecsInt", (FXint)(m_info.m_rTotalSecs));
				ReplaceFloat( i_strTemplate, "TotalMins", m_info.m_rTotalSecs / 60.f);
				ReplaceInt( i_strTemplate, "TotalMinsInt", (FXint)(m_info.m_rTotalSecs / 60.f));
				ReplaceFloat( i_strTemplate, "TotalHours", m_info.m_rTotalSecs / 3600.f);
				ReplaceInt( i_strTemplate, "TotalHoursInt", (FXint)(m_info.m_rTotalSecs / 3600.f));
				ReplaceFloat( i_strTemplate, "TotalDays", m_info.m_rTotalSecs / 86400.f);
				ReplaceInt( i_strTemplate, "TotalDaysInt", (FXint)(m_info.m_rTotalSecs / 86400.f));
				ReplaceFloat( i_strTemplate, "TotalYears", m_info.m_rTotalSecs / 31556940.f); // http://www.webexhibits.org/calendars/timeline.html 365 days 5 hrs 49 mins
				ReplaceInt( i_strTemplate, "Years", m_info.m_uYears);
				ReplaceInt( i_strTemplate, "Days", m_info.m_uDays);
				ReplaceInt( i_strTemplate, "Hours", m_info.m_uHours);
				ReplaceInt( i_strTemplate, "Mins", m_info.m_uMins);
				ReplaceInt( i_strTemplate, "Secs", m_info.m_uSecs);
				ReplaceInt( i_strTemplate, "WUs", m_info.m_uWUs);

				// Sig interval replacements
				FXint l_nInterval = m_MainSettings.readIntEntry("Sig", "Interval", k_nDefaultSigInterval);
				struct tm l_tmInterval = ConvertSecsTotmstruct(l_nInterval);
				ReplaceInt( i_strTemplate, "SigIntervalTotalSecs", l_nInterval);
				ReplaceInt( i_strTemplate, "SigIntervalDays", l_tmInterval.tm_yday);	// Based on Jan 1, 1970
				ReplaceInt( i_strTemplate, "SigIntervalHours", l_tmInterval.tm_hour);
				ReplaceInt( i_strTemplate, "SigIntervalMins", l_tmInterval.tm_min);
				ReplaceInt( i_strTemplate, "SigIntervalSecs", l_tmInterval.tm_sec);

				// HTML interval replacements
				l_nInterval = m_MainSettings.readIntEntry("HTML", "Interval", k_nDefaultHTMLInterval);
				l_tmInterval = ConvertSecsTotmstruct(l_nInterval);
				ReplaceInt( i_strTemplate, "HTMLIntervalTotalSecs", l_nInterval);
				ReplaceInt( i_strTemplate, "HTMLIntervalDays", l_tmInterval.tm_yday);	// Based on Jan 1, 1970
				ReplaceInt( i_strTemplate, "HTMLIntervalHours", l_tmInterval.tm_hour);
				ReplaceInt( i_strTemplate, "HTMLIntervalMins", l_tmInterval.tm_min);
				ReplaceInt( i_strTemplate, "HTMLIntervalSecs", l_tmInterval.tm_sec);

				// Date stuff
				ReplaceString( i_strTemplate, "CurDateSting", GetCurDateString());
				ReplaceString( i_strTemplate, "CurDateYear", GetCurDateYear());
				ReplaceString( i_strTemplate, "CurDateYr", GetCurDateYr());
				ReplaceString( i_strTemplate, "CurDateMonth", GetCurDateMonth());
				ReplaceString( i_strTemplate, "CurDateMonthNum", GetCurDateMonthNum());
				ReplaceString( i_strTemplate, "CurDateMon", GetCurDateMon());
				ReplaceString( i_strTemplate, "CurDateDayOfYear", GetCurDateDayOfYear());
				ReplaceString( i_strTemplate, "CurDateDayOfMon", GetCurDateDayOfMon());
				ReplaceString( i_strTemplate, "CurDateDayOfWeek", GetCurDateDayOfWeek());
				ReplaceString( i_strTemplate, "CurDateDayOfWk", GetCurDateDayOfWk());
				ReplaceString( i_strTemplate, "CurDateHour12", GetCurDateHour12());
				ReplaceString( i_strTemplate, "CurDateHour24", GetCurDateHour24());
				ReplaceString( i_strTemplate, "CurDateAMPM", GetCurDateAMPM());
				ReplaceString( i_strTemplate, "CurDateMinute", GetCurDateMinute());
				ReplaceString( i_strTemplate, "CurDateSecond", GetCurDateSecond());

				ReplaceString( i_strTemplate, "ProgramVersion", GetProgramVersion());
				ReplaceString( i_strTemplate, "RandomQuote", GetRandomQuote());
				}
			}
		FXFREE(&l_sTemplate);
		}

	return l_fOK;
}

//---------------------------------------------------------------------------
FXbool MightySETIInfoWindow::CreateSigString()
{
	const FXchar *lk_sTemplateFilename = m_MainSettings.readStringEntry("Sig", "Input", k_arrcDefaultSigInputName);
	FXbool l_fOK = DoReplacements(lk_sTemplateFilename, &m_strSig);

	if ( !l_fOK )
		{
		// Build default email sig if no input template
		m_strSig.format("User %s has been running %f seconds\n(%d year%s, %d day%s, %d hour%s, %d minute%s)\nand has processed %d work unit%s",
			m_info.m_strName.text(), m_info.m_rTotalSecs, m_info.m_uYears, AddS(m_info.m_uYears), m_info.m_uDays, AddS(m_info.m_uDays),
			m_info.m_uHours, AddS(m_info.m_uHours), m_info.m_uMins, AddS(m_info.m_uMins), m_info.m_uWUs, AddS(m_info.m_uWUs));

		// Output error message
		FXMessageBox::error(this,MBOX_OK,"Error Email Sig","Error reading sig template file: [%s]\nWriting default sig", lk_sTemplateFilename);
		}

	return l_fOK;
}

//---------------------------------------------------------------------------
FXbool MightySETIInfoWindow::CreateHTMLString()
{
	const FXchar *lk_sTemplateFilename = m_MainSettings.readStringEntry("HTML", "Input", k_arrcDefaultHTMLInputName);
	FXbool l_fOK = DoReplacements(lk_sTemplateFilename, &m_strHTML);

	if ( !l_fOK )
		{
		// Build default HTML if no input template
		FXString l_strTemp;
		m_strHTML.format("\
	<!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 4.0 Transitional//EN\">\n \
<HTML>\n \
	<HEAD>\n \
		<meta http-equiv=\"Content-Type\" content=\"text/html; charset=windows-1252\">\n \
		<meta http-equiv=\"Content-Language\" content=\"en-gb\">\n \
		<meta name=\"GENERATOR\" content=\"MightySETISetings v%d.%d by Drake Christensen (mighty@mightydrake.com)\">\n \
		<meta name=\"description\" content=\"SETI@Home to HTML\">\n \
		<META HTTP-EQUIV=\"Cache-Control\" CONTENT=\"no cache\">\n \
		<META HTTP-EQUIV=\"Pragma\" CONTENT=\"no cache\">\n \
		<META HTTP-EQUIV=\"Expires\" CONTENT=\"0\">\n \
		<META HTTP-EQUIV=\"Refresh\" CONTENT=\"300\">\n \
		<title>MightySETIInfo - SETI@Home Stats</title>\n \
		<base target=\"_self\">\n \
	</head>\n \
	<BODY text=#333333 bgColor=#ffffff leftMargin=0 marginheight=\"0\" marginwidth=\"0\">\n \
		User %s has been running %f seconds<br>\n \
		(%d year%s, %d day%s, %d hour%s, %d minute%s)<br>\n \
		and has processed %d work unit%s\n \
	</BODY>\n \
</HTML>\n", \

		k_uVersionMajor, k_uVersionMinor,
		m_info.m_strName.text(), m_info.m_rTotalSecs, m_info.m_uYears, AddS(m_info.m_uYears), m_info.m_uDays, AddS(m_info.m_uDays),
		m_info.m_uHours, AddS(m_info.m_uHours), m_info.m_uMins, AddS(m_info.m_uMins), m_info.m_uWUs, AddS(m_info.m_uWUs));

		// Output error message
		FXMessageBox::error(this,MBOX_OK,"Error HTML","Error reading HTML template file: [%s]", lk_sTemplateFilename);
		}

	return l_fOK;
}

//---------------------------------------------------------------------------
FXbool MightySETIInfoWindow::WriteHTMLFile()
{
	const FXchar* lk_pcHTMLFilename;
	lk_pcHTMLFilename = m_MainSettings.readStringEntry("HTML", "Output", k_arrcDefaultHTMLName);
	FXbool l_fOK = WriteFile("HTML", m_txtHTML, lk_pcHTMLFilename);
	if ( l_fOK )
		{
		WriteDateTime("HTML", "LastWritten");
		WriteINI();
		}
	return l_fOK;
}

//---------------------------------------------------------------------------
FXbool MightySETIInfoWindow::WriteSigFile()
{
	const FXchar* lk_pcSigFilename;
	lk_pcSigFilename = m_MainSettings.readStringEntry("Sig", "Output", k_arrcDefaultSigName);
	// Write to Sig file
	FXbool l_fOK = WriteFile("Sig", m_txtSig, lk_pcSigFilename);
	if ( l_fOK )
		{
		WriteDateTime("Sig", "LastWritten");
		WriteINI();
		}
	return l_fOK;
}

//---------------------------------------------------------------------------
FXbool MightySETIInfoWindow::WriteFile(const FXString i_strTitle, const SETIText *i_txtContent, const FXString i_strFilename)
{
	FXbool l_fOK = FALSE;
	FXString l_strTemp;
	// Write to output file
	l_strTemp.format("Writing [%s]", i_strTitle.text());
	Log(l_strTemp);
	l_strTemp.format("Opening output file [%s]", i_strFilename.text());
	Log(l_strTemp);
	FILE * fp=fopen(i_strFilename.text(),"w");
	if ( fp )
		{
		// Get size
		FXuint size=i_txtContent->getLength();

		FXchar * fileText;
		// Alloc buffer
		if(FXCALLOC(&fileText,FXchar,size))
			{
			// Set wait cursor
			getApp()->beginWaitCursor();

			// Get text from editor
			i_txtContent->getText(fileText,size);

			// Write the file
			l_strTemp.format("Wrinting output file [%s]", i_strFilename.text());
			Log(l_strTemp);
			FXuint n=fwrite(fileText,1,size,fp);
			FXFREE(&fileText);

			// Kill wait cursor
			getApp()->endWaitCursor();

			// Close file
			fclose(fp);

			// Were we able to write it all?
			if(n == size)
				{
				l_fOK = TRUE;
				}
			else // of write test
				{
				l_strTemp.format("Error Writing %s File", i_strTitle.text());
				FXMessageBox::error(this,MBOX_OK,l_strTemp.text(),"File: %s truncated.", i_strFilename.text());
				}
			}
		else // of open write
			{
			l_strTemp.format("Error Allocating for %s File", i_strTitle.text());
			FXMessageBox::error(this,MBOX_OK,l_strTemp.text(),"This file is too big=%ld", i_strFilename.text(), size);
			}
		}
/*
	else // of open
		{
		FXMessageBox::error(this,MBOX_OK,"Error user info file","Unable to open file: %s",i_strFilename.text());
		Log("Firing up input selection box");
		handle(this, MKUINT(ID_SETUSERINFO, SEL_COMMAND), NULL);
		}
 */
	return l_fOK;
}

//---------------------------------------------------------------------------
struct tm MightySETIInfoWindow::ConvertSecsTotmstruct(FXint i_nInterval)
{
struct tm l_tmInterval;
memset(&l_tmInterval, 0, sizeof(l_tmInterval));
#if _MSC_VER >= 1300
#error Need to change year 70 assumptions with later MS compilers
#endif
l_tmInterval.tm_year = 70;	// Whose bright idea was it to start at 70? VC98\CRT\SRC\CTIME.H(37)
l_tmInterval.tm_mday = 1;
time_t l_timeInterval = mktime(&l_tmInterval);
l_timeInterval += i_nInterval;
#if BUG_DST
// A horrible hack to fight off-by-one-hour bug in MSVC during daylight stupid time Q182042 Q190315
if ( _isindst(&l_tmInterval) )
	l_timeInterval -= 3600;
#endif
l_tmInterval = *localtime(&l_timeInterval);
return l_tmInterval;
}

//---------------------------------------------------------------------------
void MightySETIInfoWindow::WriteDateTime(FXString i_strSection, FXString i_strKey)
{
time_t l_time = time(NULL);
struct tm l_tmTime = *localtime( &l_time );
FXString l_strDate;
#if _MSC_VER >= 1300
#error Need to change year 70 assumptions with later MS compilers
#endif
l_strDate.format("%04d-%02d-%02d %02d:%02d:%02d",
	l_tmTime.tm_year + 1900, 			// Stupid, stupid, stupid \VC98\CRT\SRC\CTIME.H(37)
	l_tmTime.tm_mon + 1,
	l_tmTime.tm_mday, l_tmTime.tm_hour,
	l_tmTime.tm_min, l_tmTime.tm_sec);
m_MainSettings.writeStringEntry(i_strSection.text(), i_strKey.text(), l_strDate.text());
}

//---------------------------------------------------------------------------
void MightySETIInfoWindow::WriteINIAndDoIt(FXbool i_fExtraTest)
{
	if (WriteINI())
		{
		if ( i_fExtraTest )
			{
			ClearBoxes();
			DoIt();
			}
		}
}

//---------------------------------------------------------------------------
void MightySETIInfoWindow::DoItFirstTime()
{
Log("DoItFirstTime");
typedef enum
	{
	FinishDoNothing,
	FinishDoFTP,
	FinishDoShutdown,
	FinishDoDefibrillate
	} le_FinishUp;

// Set up to Defibrillate or shutdown
FXbool l_fOnce = m_MainSettings.readIntEntry( "System", "UpdateOnceThenExit", k_fDefaultUpdateOnceThenExit);
le_FinishUp l_FinishUp = FinishDoDefibrillate;
if ( l_fOnce )
	{
	l_FinishUp = FinishDoShutdown;
	}
if ( l_fOnce )
	{
	// We need this string whether FTP is enabled or not, so just built it once
	m_strUpdateOnceThenExit.format("Shutting down in %d seconds\nClick OK to leave program running",
		m_uDelayBeforeExit);
	}

DisplayPrefs();	// Debug output
FXbool l_fOK = CreateSigString();
// TODO: Do I want to display both sig and HTML even if OK comes back false?
if ( l_fOK )
	{
	DisplaySig();
	FXint l_fEnabled = m_MainSettings.readIntEntry("Sig", "Enabled", k_fDefaultSigEnabled);
	if ( l_fEnabled )
		{
		if ( CheckSigCountdown() )
			{
			l_fOK = WriteSigFile();
			}
		else
			{
			Log("Not time to write email sig, yet");
			}
		}
	else
		{
		Log("User chose to skip sig write");
		}
	}
if ( l_fOK )
	{
	l_fOK = CreateHTMLString();
	if ( l_fOK )
		{
		DisplayHTML();
		FXint l_fEnabled = m_MainSettings.readIntEntry("HTML", "Enabled", k_fDefaultHTMLEnabled);
		if ( l_fEnabled )
			{
			if ( CheckHTMLCountdown() )
				{
				l_fOK = WriteHTMLFile();
				if ( l_fOK )
					{
					if ( m_FTPSettings.m_fEnabled )
						{
						l_FinishUp = FinishDoFTP;
						}
					else
						{
						Log("User chose to skip FTP upload");
						if ( l_fOnce )
							{
							// TODO: Do I want to Defibrillate ahead of time and update the text in the box with countdown?
							l_FinishUp = FinishDoShutdown;
							}
						}
					}
				}
			else
				{
				Log("Not time to write HTML file, yet");
				}
			}
		else
			{
			Log("User chose to skip HTML write");
			}
		}
	}
if ( l_fOK  )
	{
	switch ( l_FinishUp )
		{
		case FinishDoNothing :
			break;
		case FinishDoFTP :
			SetFTPChore();
			break;
		case FinishDoShutdown :
			SetShutdownTimer();
			Defibrillate();
			break;
		case FinishDoDefibrillate :
			Defibrillate();
			break;
		}
	}
}

//---------------------------------------------------------------------------
void MightySETIInfoWindow::DoIt()
{
Log("User chose to do it");
KillHeartbeat();
FXbool l_fOK = CreateSigString();
if ( l_fOK )
	{
	DisplaySig();
	FXint l_fEnabled = m_MainSettings.readIntEntry("Sig", "Enabled", k_fDefaultSigEnabled);
	if ( l_fEnabled )	// Don't check countdown times if user says DoIt
		{
		l_fOK = WriteSigFile();
		}
	else
		{
		Log("User chose to skip sig write");
		}
	}
if ( l_fOK )
	{
	l_fOK = CreateHTMLString();
	if ( l_fOK )
		{
		DisplayHTML();
		FXint l_fEnabled = m_MainSettings.readIntEntry("HTML", "Enabled", k_fDefaultHTMLEnabled);
		if ( l_fEnabled )	// Don't check countdown times if user says DoIt
			{
			l_fOK = WriteHTMLFile();
			if ( l_fOK )
				{
				if ( m_FTPSettings.m_fEnabled )
					{
					SetFTPChore();
					}
				else
					{
					Log("User chose to skip FTP upload");
					}
				}
			}
		else
			{
			Log("User chose to skip HTML write");
			}
		}
	}
// This one doesn't need the switch statement like DoItFirstTime does
if ( l_fOK && !m_FTPSettings.m_fEnabled )
	{
	// If we've not set up the FTP chore then we need to defibrilate
	Defibrillate();
	}
}

//---------------------------------------------------------------------------
void MightySETIInfoWindow::Heartbeat()
{
FXbool l_fOK = TRUE;
FXbool l_fFTPing = FALSE;
FXint l_fEnabled = m_MainSettings.readIntEntry("Sig", "Enabled", k_fDefaultSigEnabled);
if ( l_fEnabled )
	{
	if ( CheckSigCountdown() )
		{
		l_fOK = CreateSigString();
		if ( l_fOK )
			{
			DisplaySig();
			l_fOK = WriteSigFile();
			}
		}
	DisplaySigTimeLeft();
	}
// Don't log disabled or countdown not expired every second

if ( l_fOK )
	{
	FXint l_fEnabled = m_MainSettings.readIntEntry("HTML", "Enabled", k_fDefaultHTMLEnabled);
	if ( l_fEnabled )
		{
		if ( CheckHTMLCountdown() )
			{
			l_fOK = CreateHTMLString();
			if ( l_fOK )
				{
				DisplayHTML();
				l_fOK = WriteHTMLFile();
				if ( l_fOK )
					{
					if ( m_FTPSettings.m_fEnabled )
						{
						SetFTPChore();
						l_fFTPing = TRUE;
						}
					}
				}
			}
		DisplayHTMLTimeLeft();
		}
	// Don't log disabled or countdown not expired every second
	}


// This one doesn't need the switch statement like DoItFirstTime does
if ( l_fOK && !l_fFTPing )
	{
	// If no errors and no FTP chore then defibrilate
	Defibrillate();
	}
}

//---------------------------------------------------------------------------
FXString MightySETIInfoWindow::MakeTimeLeftString(FXint i_nSecsLeft)
{
FXString l_strReturn;
struct tm l_tm = ConvertSecsTotmstruct(i_nSecsLeft);
l_strReturn.format("%2d Day%s, %2d Hour%s, %2d Min%s", l_tm.tm_yday, AddS(l_tm.tm_yday),
		l_tm.tm_hour, AddS(l_tm.tm_hour), l_tm.tm_min, AddS(l_tm.tm_min));
if (60 > i_nSecsLeft)
	{
	l_strReturn.format("%2d Second%s", l_tm.tm_sec, AddS(l_tm.tm_sec));
	}
return l_strReturn;
}

//---------------------------------------------------------------------------
void MightySETIInfoWindow::DisplaySigTimeLeft()
{
FXString l_strTimeLeft = MakeTimeLeftString(CalcSigTimeLeft());
m_grpboxSig->setText(FXString("Sig ") + l_strTimeLeft);
}

//---------------------------------------------------------------------------
FXbool MightySETIInfoWindow::CheckSigCountdown()
{
return CalcSigTimeLeft() <= 0;
}

//---------------------------------------------------------------------------
FXint MightySETIInfoWindow::CalcSigTimeLeft()
{
FXint l_nReturn = 0;
time_t l_CurTime = time(NULL);
struct tm l_tmLastTime = GetLastWrittenSig();
time_t l_LastTime = mktime(&l_tmLastTime);
if( (time_t)-1 == l_LastTime )
	{
	// Error converting time
	FXMessageBox::error(this,MBOX_OK,"Parsing Error","Error parsing sig last written");
	}
else
	{
#if BUG_DST
	// A horrible hack to fight off-by-one-hour bug in MSVC during daylight stupid time Q182042 Q190315
	if ( _isindst(&l_tmLastTime) )
		l_LastTime -= 3600;
#endif
	FXint l_nInterval = m_MainSettings.readIntEntry("Sig", "Interval", k_nDefaultSigInterval);
	l_nReturn = (l_LastTime + l_nInterval) - l_CurTime;
	}
return l_nReturn;
}

//---------------------------------------------------------------------------
void MightySETIInfoWindow::DisplayHTMLTimeLeft()
{
FXString l_strTimeLeft = MakeTimeLeftString(CalcHTMLTimeLeft());
m_grpboxHTML->setText(FXString("HTML ") + l_strTimeLeft);
}

//---------------------------------------------------------------------------
FXbool MightySETIInfoWindow::CheckHTMLCountdown()
{
return CalcHTMLTimeLeft() <= 0;
}

//---------------------------------------------------------------------------
FXint MightySETIInfoWindow::CalcHTMLTimeLeft()
{
FXint l_nReturn = 0;
time_t l_CurTime = time(NULL);
struct tm l_tmLastTime = GetLastWrittenHTML();
time_t l_LastTime = mktime(&l_tmLastTime);
if( (time_t)-1 == l_LastTime )
	{
	// Error converting time
	FXMessageBox::error(this,MBOX_OK,"Parsing Error","Error parsing HTML last written");
	}
else
	{
#if BUG_DST
	// A horrible hack to fight off-by-one-hour bug in MSVC during daylight stupid time Q182042 Q190315
	if ( _isindst(&l_tmLastTime) )
		l_LastTime -= 3600;
#endif
	FXint l_nInterval = m_MainSettings.readIntEntry("HTML", "Interval", k_nDefaultSigInterval);
	l_nReturn = (l_LastTime + l_nInterval) - l_CurTime;
	}
return l_nReturn;
}

//---------------------------------------------------------------------------
void MightySETIInfoWindow::Defibrillate()
{
if ( NULL == m_ptimerHeartbeat )
	{
	m_ptimerHeartbeat = getApp()->addTimeout(1000, this,ID_TIMERHEARTBEAT);
	}
}

//---------------------------------------------------------------------------
void MightySETIInfoWindow::KillHeartbeat()
{
if ( NULL != m_ptimerHeartbeat )
	{
	m_ptimerHeartbeat = getApp()->removeTimeout(m_ptimerHeartbeat);
	}
}

//---------------------------------------------------------------------------
void MightySETIInfoWindow::SetShutdownTimer()
{
	FXTimer* timerhandle = getApp()->addTimeout(m_uDelayBeforeExit * 1000, this,ID_TIMERTOSHUTDOWN);
	FXMessageBox::information(this,MBOX_OK,"Output Files Written",m_strUpdateOnceThenExit.text());
	// If user clicks okay then remove the timer so that we stay in the app
	getApp()->removeTimeout(timerhandle);
	Log("User aborted shutdown");
}

//---------------------------------------------------------------------------
//---------------------------------------------------------------------------
void MightySETIInfoWindow::SetFTPChore()
{
	// Register Chore callback message to FTP the file
	// This keeps the GUI from getting locked up while FTPing.  We get a chance to update the text boxes
	m_dlgFTPProgress = new FTPProgressDialog(this, "FTP Progress", "Transferring to FTP site\nThis may take a few seconds");
	m_dlgFTPProgress->create();
	m_dlgFTPProgress->show();
	FXChore* chorehandle = getApp()->addChore(this,ID_FTPCHORE);
}

//---------------------------------------------------------------------------
void MightySETIInfoWindow::DoFTPChore()
{
	FXbool l_fOK = TRUE;
	if ( m_FTPSettings.m_fEnabled )
		{
		// FTP it up to the server
		const FXchar * lk_pcHTMLFilename = m_MainSettings.readStringEntry("HTML", "Output", k_arrcDefaultHTMLName);
		l_fOK = FTPFile(&m_FTPSettings, lk_pcHTMLFilename);
		}
	else
		Log("User chose to skip FTP");

	m_dlgFTPProgress->onCmdOK(NULL,0,NULL);
 	delete m_dlgFTPProgress;
	m_dlgFTPProgress = NULL;

	// If everything's okay, see if we should exit
	// NOTE: If there's an error of some kind then we never start heartbeat timer, nor shutdown timer
	if ( l_fOK )
		{
		if ( m_MainSettings.readIntEntry( "System", "UpdateOnceThenExit", k_fDefaultUpdateOnceThenExit) )
			{
			SetShutdownTimer();
			Defibrillate();
			}
		else
			{
			FXTimer* timerhandle = getApp()->addTimeout(10000, this,ID_TIMERTOCONTINUE);
			Defibrillate();
  			FXGIFIcon icon(getApp(),s_infoIcon);
			m_dlgFTPContinue = new FXMessageBox(this, "FTP", "Transfer finished", &icon, MBOX_OK|DECOR_TITLE|DECOR_BORDER);
			m_dlgFTPContinue->place(PLACEMENT_OWNER);
			m_dlgFTPContinue->execute();
			getApp()->removeTimeout(timerhandle);
			delete m_dlgFTPContinue;
			m_dlgFTPContinue = NULL;
			}
		}
}

//---------------------------------------------------------------------------
void MightySETIInfoWindow::Log(FXString i_str)
{
	appendLog(i_str);
	appendLog("\n");
}

//---------------------------------------------------------------------------
void MightySETIInfoWindow::WriteDefaultINI()
{
	Log("WriteDefaultINI");
	m_MainSettings.writeStringEntry("Machines", "Root", k_arrcUserInfo);
	m_MainSettings.writeStringEntry("Sig", "Input", k_arrcDefaultSigInputName);
	m_MainSettings.writeStringEntry("Sig", "Output", k_arrcDefaultSigName);
	m_MainSettings.writeIntEntry("Sig", "Enabled", k_fDefaultSigEnabled);
	m_MainSettings.writeIntEntry("Sig", "Interval", k_nDefaultSigInterval);
	m_MainSettings.writeStringEntry("HTML", "Input", k_arrcDefaultHTMLInputName);
	m_MainSettings.writeStringEntry("HTML", "Output", k_arrcDefaultHTMLName);
	m_MainSettings.writeIntEntry("HTML", "Enabled", k_fDefaultHTMLEnabled);
	m_MainSettings.writeIntEntry("HTML", "Interval", k_nDefaultHTMLInterval);
	m_MainSettings.writeStringEntry("Quotes", "Input", k_arrcDefaultQuote);
	m_MainSettings.writeStringEntry("Quotes", "Delimiter", k_arrcQuoteDelimiter);
	m_MainSettings.writeIntEntry("Quotes", "DelimiterAdjustFront", k_nQuoteDelimiterAdjustFront);
	m_MainSettings.writeIntEntry("Quotes", "DelimiterAdjustBack", k_nQuoteDelimiterAdjustBack);
	m_MainSettings.writeIntEntry("System", "UpdateOnceThenExit", k_fDefaultUpdateOnceThenExit);
	m_MainSettings.writeIntEntry("System", "UsingSetiHide", k_fDefaultUsingSetiHide);
	WriteDefaultFTP();
	WriteINI();
}

//---------------------------------------------------------------------------
void MightySETIInfoWindow::WriteDefaultFTP()
{
	Log("WriteDefaultFTP");
	m_MainSettings.writeIntEntry("FTP", "Enabled", k_fDefaultFTPEnabled);
	m_MainSettings.writeIntEntry("FTP", "Firewall", k_nDefaultFTPFirewall);
	m_MainSettings.writeStringEntry("FTP", "Server", k_arrcDefaultFTPServer);
	m_MainSettings.writeStringEntry("FTP", "Username", k_arrcDefaultFTPUsername);
	m_MainSettings.writeStringEntry("FTP", "Password", k_arrcDefaultFTPPassword);
	m_MainSettings.writeStringEntry("FTP", "FirewallServer", k_arrcDefaultFTPFirewallServer);
	m_MainSettings.writeStringEntry("FTP", "FirewallUsername", k_arrcDefaultFTPFirewallUsername);
	m_MainSettings.writeStringEntry("FTP", "FirewallPassword", k_arrcDefaultFTPFirewallPassword);
	m_MainSettings.writeStringEntry("FTP", "Directory", k_arrcDefaultFTPDirectory);
	m_MainSettings.writeIntEntry("FTP", "PassiveMode", k_fDefaultFTPPassive);
}

//---------------------------------------------------------------------------
FXbool MightySETIInfoWindow::WriteINI()
{
	FXbool l_fReturn = TRUE;
	if ( !m_MainSettings.unparseFile(k_SettingsFilename) )
		{
		l_fReturn = FALSE;
		// UI display of error really belongs at a higher level
		FXString l_strTemp;
		FXMessageBox::error(this,MBOX_OK,"Error Saving Settings File","File: %s truncated.", k_SettingsFilename);
		l_strTemp.format("Error Saving Settings File - File: %s truncated.", k_SettingsFilename);
		Log(l_strTemp);
		}
	return l_fReturn;
}

//---------------------------------------------------------------------------
void MightySETIInfoWindow::ClearSig()
{
	m_txtSig->setText("");
}

//---------------------------------------------------------------------------
void MightySETIInfoWindow::ClearHTML()
{
	m_txtHTML->setText("");
}

//---------------------------------------------------------------------------
void MightySETIInfoWindow::ClearBoxes()
{
	Log("Clearing boxes");
	m_pdialogDebug->ClearPrefs();
	m_pdialogDebug->ClearUserInfo();
	m_pdialogDebug->ClearWorkUnitInfo();
	ClearSig();
	ClearHTML();
}

//---------------------------------------------------------------------------
void MightySETIInfoWindow::DisplaySig()
{
	m_txtSig->setText(m_strSig);
}

//---------------------------------------------------------------------------
void MightySETIInfoWindow::DisplayHTML()
{
	m_txtHTML->setText(m_strHTML);
}

//---------------------------------------------------------------------------
void MightySETIInfoWindow::DisplayPrefs()
{
	// All this just reports INI entries to prefs window in debug box
	FXString l_strFilename;
	FXString l_strTemp;

	l_strFilename = m_MainSettings.readStringEntry("Sig", "Input", k_arrcDefaultSigInputName);
	l_strTemp.format("Sig | Input=[%s]\n", l_strFilename.text());
	appendPrefs(l_strTemp);

	l_strFilename = m_MainSettings.readStringEntry("Sig", "Output", k_arrcDefaultSigName);
	l_strTemp.format("Sig | Output=[%s]\n", l_strFilename.text());
	appendPrefs(l_strTemp);

	l_strFilename = m_MainSettings.readStringEntry("HTML", "Input", k_arrcDefaultHTMLInputName);
	l_strTemp.format("HTML | Input=[%s]\n", l_strFilename.text());
	appendPrefs(l_strTemp);

	l_strFilename = m_MainSettings.readStringEntry("HTML", "Output", k_arrcDefaultHTMLName);
	l_strTemp.format("HTML | Output=[%s]\n", l_strFilename.text());
	appendPrefs(l_strTemp);
}

//---------------------------------------------------------------------------
void MightySETIInfoWindow::appendPrefs(const FXString i_str)
{
	m_pdialogDebug->AppendPrefs(i_str);
}

//---------------------------------------------------------------------------
void MightySETIInfoWindow::appendUserInfo(const FXString i_str)
{
	m_pdialogDebug->AppendUserInfo(i_str);
}

//---------------------------------------------------------------------------
void MightySETIInfoWindow::appendWorkUnit(const FXString i_str)
{
	m_pdialogDebug->AppendWorkUnitInfo(i_str);
}

//---------------------------------------------------------------------------
void MightySETIInfoWindow::appendLog(const FXString i_str)
{
	m_pdialogDebug->AppendLog(i_str);
}

//---------------------------------------------------------------------------
FXString MightySETIInfoWindow::Pluralize(const FXString i_strWord,
	const FXuint i_uNumber) const
{
	FXString l_strTemp = i_strWord;
	if ( 1 != i_uNumber )
		{
		l_strTemp += FXString("s");
		}
	return l_strTemp;
}

//---------------------------------------------------------------------------
const FXchar * MightySETIInfoWindow::AddS(const FXuint i_uNumber) const
{
	const FXchar * lk_arrTemp = k_arrcS;
	if ( 1 == i_uNumber )
		{
		lk_arrTemp = k_arrcEmpty;
		}
	return lk_arrTemp;
}

//---------------------------------------------------------------------------
struct tm MightySETIInfoWindow::ParseATime(FXString i_strTime,
		const FXchar * i_arrcDefault)
{
// Very strict test of format YYYY-MM-DD HH:MM:SS
//                            0123456789012345678
if ( !((i_strTime[4] == '-') &&
	(i_strTime[7] == '-') &&
	(i_strTime[10] == ' ') &&
	(i_strTime[13] == ':') &&
	(i_strTime[16] == ':') &&
	(i_strTime.length() == 19)
	 ))
	{
	i_strTime = i_arrcDefault;
	}

struct tm l_tmTime;
memset(&l_tmTime, 0, sizeof(l_tmTime));
#if _MSC_VER >= 1300
#error Need to change year 70 assumptions with later MS compilers
#endif
l_tmTime.tm_year = atoi((i_strTime.mid(0, 4)).text()) - 1900;	// Whose bright idea was it to start at 70? VC98\CRT\SRC\CTIME.H(37)
l_tmTime.tm_mon = atoi((i_strTime.mid(5, 2)).text()) - 1;
l_tmTime.tm_mday = atoi((i_strTime.mid(8, 2)).text());
l_tmTime.tm_hour = atoi((i_strTime.mid(11, 2)).text());
l_tmTime.tm_min = atoi((i_strTime.mid(14, 2)).text());
l_tmTime.tm_sec = atoi((i_strTime.mid(17, 2)).text());
return l_tmTime;
}

//---------------------------------------------------------------------------
struct tm MightySETIInfoWindow::GetLastWrittenSig()
{
FXString l_strTime = m_MainSettings.readStringEntry("Sig", "LastWritten",
		k_arrcDefaultSigLastWritten);
struct tm l_tmTime = ParseATime(l_strTime, k_arrcDefaultSigLastWritten);
return l_tmTime;
}

//---------------------------------------------------------------------------
struct tm MightySETIInfoWindow::GetLastWrittenHTML()
{
FXString l_strTime = m_MainSettings.readStringEntry("HTML", "LastWritten",
		k_arrcDefaultSigLastWritten);
struct tm l_tmTime = ParseATime(l_strTime, k_arrcDefaultHTMLLastWritten);
return l_tmTime;
}

//---------------------------------------------------------------------------
FXString MightySETIInfoWindow::GetCurDateString()
{
	// MSDN localtime example
	struct tm *l_time;

	l_time = localtime( &m_info.m_time ); /* Convert to local time. */
	return FXString(asctime(l_time));
}

//---------------------------------------------------------------------------
FXString MightySETIInfoWindow::GetCurDateYear()
{
	// MSDN strftime example
	struct tm *l_time;
	FXchar l_sTemp[128];

	l_time = localtime( &m_info.m_time ); /* Convert to local time. */
	strftime( l_sTemp, 128, "%Y", l_time );

	return FXString(l_sTemp);
}

//---------------------------------------------------------------------------
FXString MightySETIInfoWindow::GetCurDateYr()
{
	// MSDN strftime example
	struct tm *l_time;
	FXchar l_sTemp[128];

	l_time = localtime( &m_info.m_time ); /* Convert to local time. */
	strftime( l_sTemp, 128, "%y", l_time );

	return FXString(l_sTemp);
}

//---------------------------------------------------------------------------
FXString MightySETIInfoWindow::GetCurDateMonth()
{
	// MSDN strftime example
	struct tm *l_time;
	FXchar l_sTemp[128];

	l_time = localtime( &m_info.m_time ); /* Convert to local time. */
	strftime( l_sTemp, 128, "%B", l_time );

	return FXString(l_sTemp);
}

//---------------------------------------------------------------------------
FXString MightySETIInfoWindow::GetCurDateMon()
{
	// MSDN strftime example
	struct tm *l_time;
	FXchar l_sTemp[128];

	l_time = localtime( &m_info.m_time ); /* Convert to local time. */
	strftime( l_sTemp, 128, "%b", l_time );

	return FXString(l_sTemp);
}

//---------------------------------------------------------------------------
FXString MightySETIInfoWindow::GetCurDateMonthNum()
{
	// MSDN strftime example
	struct tm *l_time;
	FXchar l_sTemp[128];

	l_time = localtime( &m_info.m_time ); /* Convert to local time. */
	strftime( l_sTemp, 128, "%m", l_time );

	return FXString(l_sTemp);
}

//---------------------------------------------------------------------------
FXString MightySETIInfoWindow::GetCurDateDayOfYear()
{
	// MSDN strftime example
	struct tm *l_time;
	FXchar l_sTemp[128];

	l_time = localtime( &m_info.m_time ); /* Convert to local time. */
	strftime( l_sTemp, 128, "%j", l_time );

	return FXString(l_sTemp);
}

//---------------------------------------------------------------------------
FXString MightySETIInfoWindow::GetCurDateDayOfMon()
{
	// MSDN strftime example
	struct tm *l_time;
	FXchar l_sTemp[128];

	l_time = localtime( &m_info.m_time ); /* Convert to local time. */
	strftime( l_sTemp, 128, "%d", l_time );

	return FXString(l_sTemp);
}

//---------------------------------------------------------------------------
FXString MightySETIInfoWindow::GetCurDateDayOfWeek()
{
	// MSDN strftime example
	struct tm *l_time;
	FXchar l_sTemp[128];

	l_time = localtime( &m_info.m_time ); /* Convert to local time. */
	strftime( l_sTemp, 128, "%A", l_time );

	return FXString(l_sTemp);
}

//---------------------------------------------------------------------------
FXString MightySETIInfoWindow::GetCurDateDayOfWk()
{
	// MSDN strftime example
	struct tm *l_time;
	FXchar l_sTemp[128];

	l_time = localtime( &m_info.m_time ); /* Convert to local time. */
	strftime( l_sTemp, 128, "%a", l_time );

	return FXString(l_sTemp);
}

//---------------------------------------------------------------------------
FXString MightySETIInfoWindow::GetCurDateHour12()
{
	// MSDN strftime example
	struct tm *l_time;
	FXchar l_sTemp[128];

	l_time = localtime( &m_info.m_time ); /* Convert to local time. */
	strftime( l_sTemp, 128, "%I", l_time );

	return FXString(l_sTemp);
}

//---------------------------------------------------------------------------
FXString MightySETIInfoWindow::GetCurDateHour24()
{
	// MSDN strftime example
	struct tm *l_time;
	FXchar l_sTemp[128];

	l_time = localtime( &m_info.m_time ); /* Convert to local time. */
	strftime( l_sTemp, 128, "%H", l_time );

	return FXString(l_sTemp);
}

//---------------------------------------------------------------------------
FXString MightySETIInfoWindow::GetCurDateAMPM()
{
	// MSDN strftime example
	struct tm *l_time;
	FXchar l_sTemp[128];

	l_time = localtime( &m_info.m_time ); /* Convert to local time. */
	strftime( l_sTemp, 128, "%p", l_time );

	return FXString(l_sTemp);
}

//---------------------------------------------------------------------------
FXString MightySETIInfoWindow::GetCurDateMinute()
{
	// MSDN strftime example
	struct tm *l_time;
	FXchar l_sTemp[128];

	l_time = localtime( &m_info.m_time ); /* Convert to local time. */
	strftime( l_sTemp, 128, "%M", l_time );

	return FXString(l_sTemp);
}

//---------------------------------------------------------------------------
FXString MightySETIInfoWindow::GetCurDateSecond()
{
	// MSDN strftime example
	struct tm *l_time;
	FXchar l_sTemp[128];

	l_time = localtime( &m_info.m_time ); /* Convert to local time. */
	strftime( l_sTemp, 128, "%S", l_time );

	return FXString(l_sTemp);
}

//---------------------------------------------------------------------------
FXString MightySETIInfoWindow::GetProgramVersion()
{
FXString l_strTemp;
l_strTemp.format("v%d.%d %s", k_uVersionMajor, k_uVersionMinor, k_arrcVersionModifier);
return l_strTemp;
}

//---------------------------------------------------------------------------
FXString MightySETIInfoWindow::GetRandomQuote()
{
FXString l_strReturn;
FXString l_strFilename = m_MainSettings.readStringEntry("Quotes", "Input", k_arrcDefaultQuote);
FXbool l_fOK = FALSE;
FILE * fp=fopen(l_strFilename.text(),"rb");
if(fp)
	{
	// Read input file
	FXuint l_iSize=FXFile::size(l_strFilename);

	// Make buffer to load file
	FXchar *l_sQuotes;
	FXbool l_fAlloced = FXCALLOC(&l_sQuotes,FXchar,l_iSize+1);
	if (l_fAlloced)
		{
		// Read the file
		FXuint n=fread(l_sQuotes,1,l_iSize,fp);
		// Close file
		fclose(fp);
		// For now, the quotes file is reopened and reread just like the template files.
		memset(l_sQuotes, 0, l_iSize);
		fp=fopen(l_strFilename.text(),"r");
		FXint l_nRead = fread(l_sQuotes,1,l_iSize,fp);
		fclose(fp);

		if ( n == l_iSize)
			{
			l_fOK = TRUE;
   			FXString l_strDelimiter = m_MainSettings.readStringEntry("Quotes", "Delimiter", k_arrcQuoteDelimiter);
			FXString l_strTemp = l_strDelimiter; //FXString("^") + l_strDelimiter + FXString("$");
			// There are two ways to do this
			// One is to pre-parse the file to find out how many quote entries there are
			//   and choose the random number based on that
			// The other is to offset randomly into the file and then search backward and
			//   forward from that spot for the delimiters
			// I'm using the second for now
			float l_fRandom = (float)rand() / (float)RAND_MAX;
			FXint l_nRandPos = (FXint)((float)l_nRead * l_fRandom);

			FXRex l_rex;
			FXRexError l_rexErr;
			FXint const lk_iComparisons = 1;
			l_rexErr = l_rex.parse(l_strTemp, REX_ICASE);
			if ( REGERR_OK == l_rexErr )
				{
				FXint l_nBeginning1 = 0;
				FXint l_nEnd1 = 0;
				FXint l_nBeginning2 = 0;
				FXint l_nEnd2 = 0;
				FXbool l_fOK = l_rex.match(l_sQuotes, &l_nBeginning1, &l_nEnd1, REX_BACKWARD, lk_iComparisons, 0, l_nRandPos);
				if ( !l_fOK )
					{
					// If no hit then assume we were at the beginning of the file and there was no delimiter
					l_nBeginning1 = l_nEnd1 = 0;
					}

   				FXint l_nDelimiterAdjustFront = m_MainSettings.readIntEntry("Quotes", "DelimiterAdjustFront", k_nQuoteDelimiterAdjustFront);
				l_fOK = l_rex.match(l_sQuotes, &l_nBeginning2, &l_nEnd2, REX_FORWARD, lk_iComparisons,
						l_nEnd1 + l_nDelimiterAdjustFront, l_nRead);
				if ( l_fOK )
					{
   					FXint l_nDelimiterAdjustBack = m_MainSettings.readIntEntry("Quotes", "DelimiterAdjustBack", k_nQuoteDelimiterAdjustBack);
					FXint l_nLength = (l_nBeginning2 - l_nEnd1) - (l_nDelimiterAdjustFront + l_nDelimiterAdjustBack); // Trying to strip delimiter from quote
					FXchar * l_sQuote = NULL;
					FXCALLOC(&l_sQuote, FXchar, l_nLength+1);
					memset(l_sQuote, 0, l_nLength);
					strncpy(l_sQuote, &(l_sQuotes[l_nEnd1+l_nDelimiterAdjustFront]), l_nLength);
					l_strReturn = l_sQuote;
					FXFREE(&l_sQuote);
					}
				else
					{
					l_strReturn = "Unable to find quote delimiter";
					}
				}
			else
				{
				Log(FXRex::getError(l_rexErr));
				}
			}
		}
	FXFREE(&l_sQuotes);
	}

return l_strReturn;
}

//---------------------------------------------------------------------------
void MightySETIInfoWindow::GetFTPSettings(FTPSettings * i_pFTPSettings)
{
	i_pFTPSettings->m_fEnabled = m_MainSettings.readIntEntry("FTP", "Enabled", k_fDefaultFTPEnabled);
	i_pFTPSettings->m_nFirewallIndex = m_MainSettings.readIntEntry("FTP", "Firewall", k_nDefaultFTPFirewall);
	i_pFTPSettings->m_strServer = m_MainSettings.readStringEntry("FTP", "Server", k_arrcDefaultFTPServer);
	i_pFTPSettings->m_strUsername = m_MainSettings.readStringEntry("FTP", "Username", k_arrcDefaultFTPUsername);
	i_pFTPSettings->m_strPassword = m_MainSettings.readStringEntry("FTP", "Password", k_arrcDefaultFTPPassword);
	i_pFTPSettings->m_strDirectory = m_MainSettings.readStringEntry("FTP", "Directory", k_arrcDefaultFTPDirectory);
	i_pFTPSettings->m_strFirewallServer = m_MainSettings.readStringEntry("FTP", "FirewallServer", k_arrcDefaultFTPFirewallServer);
	i_pFTPSettings->m_strFirewallUsername = m_MainSettings.readStringEntry("FTP", "FirewallUsername", k_arrcDefaultFTPFirewallUsername);
	i_pFTPSettings->m_strFirewallPassword = m_MainSettings.readStringEntry("FTP", "FirewallPassword", k_arrcDefaultFTPFirewallPassword);
	i_pFTPSettings->m_fPassive = m_MainSettings.readIntEntry("FTP", "PassiveMode", k_fDefaultFTPPassive);
}

//---------------------------------------------------------------------------
void MightySETIInfoWindow::SaveFTPSettings(FTPSettings * i_pFTPSettings)
{
	m_MainSettings.writeIntEntry("FTP", "Enabled", i_pFTPSettings->m_fEnabled);
	m_MainSettings.writeIntEntry("FTP", "Firewall", i_pFTPSettings->m_nFirewallIndex);
	m_MainSettings.writeStringEntry("FTP", "Server", i_pFTPSettings->m_strServer.text());
	m_MainSettings.writeStringEntry("FTP", "Username", i_pFTPSettings->m_strUsername.text());
	m_MainSettings.writeStringEntry("FTP", "Password", i_pFTPSettings->m_strPassword.text());
	m_MainSettings.writeStringEntry("FTP", "FirewallServer", i_pFTPSettings->m_strFirewallServer.text());
	m_MainSettings.writeStringEntry("FTP", "FirewallUsername", i_pFTPSettings->m_strFirewallUsername.text());
	m_MainSettings.writeStringEntry("FTP", "FirewallPassword", i_pFTPSettings->m_strFirewallPassword.text());
	m_MainSettings.writeStringEntry("FTP", "Directory", i_pFTPSettings->m_strDirectory.text());
	m_MainSettings.writeIntEntry("FTP", "PassiveMode",i_pFTPSettings->m_fPassive);
}

//---------------------------------------------------------------------------
FXbool MightySETIInfoWindow::FTPFile(const FTPSettings *i_pFTPSettings, const FXString i_strFilename)
{
	FXString l_strTemp;
	SETIFTP * l_pFTP = InstantiateFTP();
	FXint l_nError = SETIFTPERR_OK;

    /* initialize FTP sesson */
	l_strTemp.format("Initializing FTP - Passive = %d", i_pFTPSettings->m_fPassive);
	Log(l_strTemp);
	l_nError = l_pFTP->Init(i_pFTPSettings->m_fPassive);

	if ( SETIFTPERR_OK == l_nError )
		{
	    l_strTemp.format("--- Connection on %s  ---", i_pFTPSettings->m_strServer.text() );
		Log(l_strTemp);
		l_nError = l_pFTP->Login ( i_pFTPSettings->m_strServer,
		                            i_pFTPSettings->m_strUsername,
		                            i_pFTPSettings->m_strPassword,
		                            i_pFTPSettings->m_strFirewallServer,
		                            i_pFTPSettings->m_strFirewallUsername,
		                            i_pFTPSettings->m_strFirewallPassword,
		                            i_pFTPSettings->m_nFirewallIndex);
		l_strTemp.format("Login returns %d", l_nError);
		Log(l_strTemp);

		if ( SETIFTPERR_OK == l_nError )
			{
			FXString l_strOutput;
			l_strOutput = i_pFTPSettings->m_strDirectory + FXFile::name(i_strFilename);
			l_strTemp.format("Send file [%s] to [%s]", i_strFilename.text(), l_strOutput.text());
			Log(l_strTemp);
		    l_nError = l_pFTP->SendFile (  i_strFilename.text(),
		                        l_strOutput.text());
			l_strTemp.format("Send returns %d [%s]", l_nError, l_pFTP->ErrorToString(l_nError) );
			Log(l_strTemp);

			l_nError = l_pFTP->Disconnect();
			}
		}

	// Catch-all for error messages
	if ( SETIFTPERR_OK != l_nError )
		{
		FXMessageBox::error(this,MBOX_OK,"Error Initializing FTP Session",l_pFTP->ErrorToString(l_nError).text());
		}

	l_pFTP->Shutdown();

	delete l_pFTP;

	return SETIFTPERR_OK == l_nError;
}

//---------------------------------------------------------------------------
// Start the whole thing
int main(int argc,char *argv[]){

  // Make application
  MightySETIInfoApp application("MightySETIInfo","Mighty");

  // Open display
  application.init(argc,argv);

  // Make window
  new MightySETIInfoWindow(&application);

  // Create app
  application.create();

  // Run
  return application.run();
  }


